<?php
/**
 * ============================================================
 * JSON RESPONSE HELPER
 * ============================================================
 * Standardized JSON responses for all AJAX endpoints
 */

class Response {
    /**
     * Send success response
     * @param string $message
     * @param array $data Additional data to include
     * @param int $statusCode HTTP status code
     */
    public static function success(string $message = 'Success', array $data = [], int $statusCode = 200): void {
        self::json([
            'success' => true,
            'message' => $message,
            'data'    => $data
        ], $statusCode);
    }
    
    /**
     * Send error response
     * @param string $message
     * @param array $errors Validation errors or additional error info
     * @param int $statusCode HTTP status code
     */
    public static function error(string $message = 'Error', array $errors = [], int $statusCode = 400): void {
        self::json([
            'success' => false,
            'message' => $message,
            'errors'  => $errors
        ], $statusCode);
    }
    
    /**
     * Send validation error response
     * @param array $errors Field-specific errors
     * @param string $message
     */
    public static function validationError(array $errors, string $message = 'অনুগ্রহ করে সকল তথ্য সঠিকভাবে পূরণ করুন'): void {
        self::json([
            'success' => false,
            'message' => $message,
            'errors'  => $errors
        ], 422);
    }
    
    /**
     * Send unauthorized response
     * @param string $message
     */
    public static function unauthorized(string $message = 'অনুগ্রহ করে লগইন করুন'): void {
        self::json([
            'success' => false,
            'message' => $message
        ], 401);
    }
    
    /**
     * Send forbidden response
     * @param string $message
     */
    public static function forbidden(string $message = 'আপনার এই কাজ করার অনুমতি নেই'): void {
        self::json([
            'success' => false,
            'message' => $message
        ], 403);
    }
    
    /**
     * Send not found response
     * @param string $message
     */
    public static function notFound(string $message = 'তথ্য পাওয়া যায়নি'): void {
        self::json([
            'success' => false,
            'message' => $message
        ], 404);
    }
    
    /**
     * Send CSRF error response
     */
    public static function csrfError(): void {
        self::json([
            'success' => false,
            'message' => 'নিরাপত্তা টোকেন অবৈধ। পেজ রিফ্রেশ করে আবার চেষ্টা করুন।'
        ], 403);
    }
    
    /**
     * Send server error response
     * @param string $message
     */
    public static function serverError(string $message = 'সার্ভার ত্রুটি। অনুগ্রহ করে পরে আবার চেষ্টা করুন।'): void {
        self::json([
            'success' => false,
            'message' => $message
        ], 500);
    }
    
    /**
     * Send raw JSON response
     * @param array $data
     * @param int $statusCode
     */
    public static function json(array $data, int $statusCode = 200): void {
        // Clean any previous output
        if (ob_get_level()) {
            ob_clean();
        }
        
        // Set headers
        http_response_code($statusCode);
        header('Content-Type: application/json; charset=utf-8');
        header('Cache-Control: no-cache, no-store, must-revalidate');
        header('Pragma: no-cache');
        header('Expires: 0');
        
        // Output JSON
        echo json_encode($data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        exit;
    }
}

/**
 * Shorthand for success response
 */
function jsonSuccess(string $message = 'Success', array $data = []): void {
    Response::success($message, $data);
}

/**
 * Shorthand for error response
 */
function jsonError(string $message = 'Error', array $errors = [], int $code = 400): void {
    Response::error($message, $errors, $code);
}
