/**
 * ============================================================
 * SAFA MODEST HOUSE - Global AJAX Configuration
 * ============================================================
 * Sets up jQuery AJAX defaults, CSRF handling, and global behaviors
 */

(function($) {
    'use strict';
    
    // ============================================================
    // CSRF Token Setup
    // ============================================================
    const csrfToken = $('meta[name="csrf-token"]').attr('content');
    
    // Add CSRF token to all AJAX requests
    $.ajaxSetup({
        headers: {
            'X-CSRF-TOKEN': csrfToken,
            'X-Requested-With': 'XMLHttpRequest'
        }
    });
    
    // ============================================================
    // Global AJAX Events
    // ============================================================
    
    // Show loader on AJAX start
    $(document).ajaxStart(function() {
        $('#global-loader').removeClass('d-none');
    });
    
    // Hide loader on AJAX complete
    $(document).ajaxStop(function() {
        $('#global-loader').addClass('d-none');
    });
    
    // Global AJAX error handler
    $(document).ajaxError(function(event, jqXHR, settings, thrownError) {
        // Don't show error for aborted requests
        if (thrownError === 'abort') return;
        
        let message = 'একটি সমস্যা হয়েছে। অনুগ্রহ করে আবার চেষ্টা করুন।';
        
        if (jqXHR.status === 401) {
            message = 'অনুগ্রহ করে লগইন করুন।';
            // Redirect to login after 2 seconds
            setTimeout(function() {
                window.location.href = window.siteUrl + '/public/?page=login';
            }, 2000);
        } else if (jqXHR.status === 403) {
            message = 'আপনার এই কাজ করার অনুমতি নেই।';
        } else if (jqXHR.status === 422) {
            // Validation error - let individual handlers deal with it
            return;
        } else if (jqXHR.status === 500) {
            message = 'সার্ভার ত্রুটি। অনুগ্রহ করে পরে আবার চেষ্টা করুন।';
        }
        
        // Show error toast
        showToast(message, 'error');
    });
    
    // ============================================================
    // Toast Notification Helper
    // ============================================================
    window.showToast = function(message, type = 'success', duration = 3000) {
        const icons = {
            success: 'success',
            error: 'error',
            warning: 'warning',
            info: 'info'
        };
        
        Swal.fire({
            toast: true,
            position: 'top-end',
            icon: icons[type] || 'info',
            title: message,
            showConfirmButton: false,
            timer: duration,
            timerProgressBar: true,
            didOpen: (toast) => {
                toast.addEventListener('mouseenter', Swal.stopTimer);
                toast.addEventListener('mouseleave', Swal.resumeTimer);
            }
        });
    };
    
    // ============================================================
    // Confirm Dialog Helper
    // ============================================================
    window.confirmAction = function(options = {}) {
        const defaults = {
            title: 'আপনি কি নিশ্চিত?',
            text: 'এই কাজটি পূর্বাবস্থায় ফেরানো যাবে না।',
            icon: 'warning',
            confirmButtonText: 'হ্যাঁ, করুন',
            cancelButtonText: 'বাতিল',
            confirmButtonColor: '#dc3545',
            showCancelButton: true
        };
        
        return Swal.fire({ ...defaults, ...options });
    };
    
    // ============================================================
    // Format Price Helper
    // ============================================================
    window.formatPrice = function(amount) {
        return '৳ ' + parseFloat(amount).toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,');
    };
    
    // ============================================================
    // Update Cart Count
    // ============================================================
    window.updateCartCount = function(count) {
        const $badge = $('.cart-count');
        if (count > 0) {
            $badge.text(count).show();
        } else {
            $badge.hide();
        }
    };
    
    // ============================================================
    // Form Validation Helper
    // ============================================================
    window.clearFormErrors = function($form) {
        $form.find('.is-invalid').removeClass('is-invalid');
        $form.find('.invalid-feedback').remove();
    };
    
    window.showFormErrors = function($form, errors) {
        clearFormErrors($form);
        
        $.each(errors, function(field, message) {
            const $field = $form.find('[name="' + field + '"]');
            $field.addClass('is-invalid');
            $field.after('<div class="invalid-feedback">' + message + '</div>');
        });
    };
    
    // ============================================================
    // Disable/Enable Submit Button
    // ============================================================
    window.setButtonLoading = function($btn, loading = true) {
        if (loading) {
            $btn.data('original-text', $btn.html());
            $btn.html('<span class="spinner-border spinner-border-sm me-2"></span>অপেক্ষা করুন...');
            $btn.prop('disabled', true);
        } else {
            $btn.html($btn.data('original-text'));
            $btn.prop('disabled', false);
        }
    };
    
})(jQuery);
