<?php
/**
 * ============================================================
 * PRODUCTS LISTING PAGE
 * ============================================================
 */

require_once APP_PATH . '/services/product_service.php';

// Get filters from query string
$filters = [];
$categorySlug = sanitize($_GET['category'] ?? '');
$search = sanitize($_GET['search'] ?? '');
$sort = sanitize($_GET['sort'] ?? 'newest');
$featured = isset($_GET['featured']);

if ($categorySlug) {
    $filters['category_slug'] = $categorySlug;
}
if ($search) {
    $filters['search'] = $search;
}
if ($featured) {
    $filters['featured'] = true;
}
$filters['sort'] = $sort;

// Pagination
$page = max(1, (int)($_GET['p'] ?? 1));
$limit = PRODUCTS_PER_PAGE;
$offset = ($page - 1) * $limit;

// Get products
$products = ProductService::getAll($filters, $limit, $offset);
$totalProducts = ProductService::getCount($filters);
$totalPages = ceil($totalProducts / $limit);

// Get categories for filter
$categories = Database::fetchAll(
    "SELECT id, name, slug FROM categories WHERE is_active = 1 ORDER BY sort_order"
);

// Get current category
$currentCategory = null;
if ($categorySlug) {
    $currentCategory = Database::fetchOne("SELECT * FROM categories WHERE slug = ?", [$categorySlug]);
}

// Page meta
$pageTitle = $currentCategory ? $currentCategory['name'] : ($search ? "অনুসন্ধান: $search" : 'সকল পণ্য');
$pageDescription = 'সেরা মানের পোশাক ও আনুষাঙ্গিক সামগ্রী';

include PUBLIC_PATH . '/includes/head.php';
include PUBLIC_PATH . '/includes/navbar.php';
?>

<main class="py-4">
    <div class="container">
        <!-- Breadcrumb -->
        <nav aria-label="breadcrumb" class="mb-4">
            <ol class="breadcrumb">
                <li class="breadcrumb-item"><a href="<?= url('public/') ?>">হোম</a></li>
                <?php if ($currentCategory): ?>
                <li class="breadcrumb-item"><a href="<?= url('public/?page=products') ?>">পণ্য</a></li>
                <li class="breadcrumb-item active"><?= sanitize($currentCategory['name']) ?></li>
                <?php else: ?>
                <li class="breadcrumb-item active">পণ্য</li>
                <?php endif; ?>
            </ol>
        </nav>
        
        <div class="row">
            <!-- Sidebar Filters -->
            <div class="col-lg-3 mb-4">
                <div class="bg-white rounded p-3 shadow-sm">
                    <h5 class="mb-3">ক্যাটাগরি</h5>
                    <ul class="list-unstyled">
                        <li class="mb-2">
                            <a href="<?= url('public/?page=products') ?>" 
                               class="text-decoration-none <?= !$categorySlug ? 'text-primary fw-bold' : 'text-dark' ?>">
                                সকল পণ্য
                            </a>
                        </li>
                        <?php foreach ($categories as $cat): ?>
                        <li class="mb-2">
                            <a href="<?= url('public/?page=products&category=' . $cat['slug']) ?>" 
                               class="text-decoration-none <?= $categorySlug === $cat['slug'] ? 'text-primary fw-bold' : 'text-dark' ?>">
                                <?= sanitize($cat['name']) ?>
                            </a>
                        </li>
                        <?php endforeach; ?>
                    </ul>
                </div>
            </div>
            
            <!-- Products Grid -->
            <div class="col-lg-9">
                <!-- Header -->
                <div class="d-flex justify-content-between align-items-center mb-4">
                    <div>
                        <h1 class="h4 mb-0"><?= sanitize($pageTitle) ?></h1>
                        <small class="text-muted"><?= $totalProducts ?> পণ্য পাওয়া গেছে</small>
                    </div>
                    
                    <!-- Sort Dropdown -->
                    <div class="dropdown">
                        <button class="btn btn-outline-secondary dropdown-toggle" type="button" data-bs-toggle="dropdown">
                            <i class="bi bi-sort-down me-1"></i>
                            <?php
                            $sortLabels = [
                                'newest' => 'নতুন',
                                'popular' => 'জনপ্রিয়',
                                'price_low' => 'দাম: কম → বেশি',
                                'price_high' => 'দাম: বেশি → কম',
                                'rating' => 'রেটিং'
                            ];
                            echo $sortLabels[$sort] ?? 'সাজান';
                            ?>
                        </button>
                        <ul class="dropdown-menu dropdown-menu-end">
                            <?php foreach ($sortLabels as $key => $label): ?>
                            <li>
                                <a class="dropdown-item <?= $sort === $key ? 'active' : '' ?>" 
                                   href="<?= url('public/?page=products' . ($categorySlug ? '&category=' . $categorySlug : '') . '&sort=' . $key) ?>">
                                    <?= $label ?>
                                </a>
                            </li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                </div>
                
                <?php if (empty($products)): ?>
                <!-- No Products -->
                <div class="text-center py-5">
                    <i class="bi bi-box-seam display-1 text-muted"></i>
                    <h4 class="mt-3">কোনো পণ্য পাওয়া যায়নি</h4>
                    <p class="text-muted">অন্য ক্যাটাগরি দেখুন অথবা পরে আবার চেষ্টা করুন।</p>
                    <a href="<?= url('public/?page=products') ?>" class="btn btn-primary">সকল পণ্য দেখুন</a>
                </div>
                <?php else: ?>
                
                <!-- Products Grid -->
                <div class="row g-4">
                    <?php foreach ($products as $product): ?>
                    <div class="col-6 col-md-4">
                        <?php include PUBLIC_PATH . '/includes/product-card.php'; ?>
                    </div>
                    <?php endforeach; ?>
                </div>
                
                <!-- Pagination -->
                <?php if ($totalPages > 1): ?>
                <nav class="mt-5">
                    <ul class="pagination justify-content-center">
                        <?php if ($page > 1): ?>
                        <li class="page-item">
                            <a class="page-link" href="<?= url('public/?page=products' . ($categorySlug ? '&category=' . $categorySlug : '') . '&sort=' . $sort . '&p=' . ($page - 1)) ?>">
                                <i class="bi bi-chevron-left"></i>
                            </a>
                        </li>
                        <?php endif; ?>
                        
                        <?php for ($i = 1; $i <= $totalPages; $i++): ?>
                        <li class="page-item <?= $i === $page ? 'active' : '' ?>">
                            <a class="page-link" href="<?= url('public/?page=products' . ($categorySlug ? '&category=' . $categorySlug : '') . '&sort=' . $sort . '&p=' . $i) ?>">
                                <?= $i ?>
                            </a>
                        </li>
                        <?php endfor; ?>
                        
                        <?php if ($page < $totalPages): ?>
                        <li class="page-item">
                            <a class="page-link" href="<?= url('public/?page=products' . ($categorySlug ? '&category=' . $categorySlug : '') . '&sort=' . $sort . '&p=' . ($page + 1)) ?>">
                                <i class="bi bi-chevron-right"></i>
                            </a>
                        </li>
                        <?php endif; ?>
                    </ul>
                </nav>
                <?php endif; ?>
                
                <?php endif; ?>
            </div>
        </div>
    </div>
</main>

<?php include PUBLIC_PATH . '/includes/footer.php'; ?>
